#include "all_max.h"
#include "input.h"
#include "jacobi.h"
#include <cmath>

class ggraph {
  vector<vector<int>> edge;

public:
  ggraph();
  ggraph(symmetric_matrix &);

  int rows() const;
  int &operator()(const int, const int);
  const vector<int> &row(const int) const;

  void input(istream &);
  void input(const vector<int> &);
  void input(int i);
  void new_row();
  void output(ostream &) const;
};

ggraph::ggraph() {}

ggraph::ggraph(symmetric_matrix &m) {
  unsigned long int i, j;

  for (i = 1; i <= m.order(); i++) {
    edge.push_back(vector<int>());

    for (j = 1; j <= m.order(); j++)
      edge[i - 1].push_back(fabs(m(i, j)) >= EPS);
  }
}

int ggraph::rows() const { return edge.size(); }

istream &operator>>(istream &istr, vector<int> &v) {
  char c;

  v = vector<int>();

  while (true) {
    istr.get(c);

    if (istr.eof() || c == '\n')
      break;

    if (isdigit(c)) {
      int value = 0;

      do {
        value = value * 10 + (c - '0');

        istr.get(c);

        if (istr.eof())
          break;
      } while (isdigit(c));

      v.push_back(value);

      if (istr.eof() || c == '\n')
        break;
    }
  }

  return istr;
}

ostream &operator<<(ostream &ostr, vector<int> &v) {
  int i;

  for (i = 0; i < v.size(); i++) {
    if (i != 0)
      ostr << ' ';

    ostr << v[i];
  }

  return ostr;
}

int &ggraph::operator()(const int i, const int j) {
  if (edge.size() == 0)
    throw(string("Warning: Access to the edge of unformed graph!"));

  if (edge[edge.size() - 1].size() != edge.size())
    throw(string("Warning: Access to the edge of unformed graph!"));

  if (i < 1 || j < 1 || i > edge.size() || j > edge[0].size())
    throw(string(
        "Warning: Endvertices of the corresponding edge are out of range!"));

  return edge[i - 1][j - 1];
}

const vector<int> &ggraph::row(const int i) const {
  if (edge.size() == 0)
    throw(string("Warning: Access to the nonformed matrix row!"));

  if (i < 1 || i > edge.size())
    throw(string("Warning: Row index is out of range!"));

  return edge[i - 1];
}

void ggraph::input(istream &istr) {
  vector<int> row;

  edge = vector<vector<int>>();

  istr >> row;

  if (row.empty())
    throw(string("Warning: There is no graph matrix!"));

  while (true) {
    edge.push_back(row);

    istr >> row;
    if (row.size() == 0)
      break;

    if (row.size() != edge[0].size())
      throw(string("Warning: Illegal entries!"));

    if (row.size() <= edge.size())
      throw(string("Warning: Illegal entries!"));
  }

  if (edge[0].size() != edge.size())
    throw(string("Warning: Illegal entries!"));
}

void ggraph::input(int i) {
  int last;

  if (edge.size() == 0) {
    edge.push_back(vector<int>());
    edge[0].push_back(i);

    return;
  }

  last = edge.size() - 1;

  if (last > 0)
    if (edge[last].size() == edge[0].size())
      if (last == edge[0].size() - 1)
        throw(string("Warning: Illegal entries!"));
      else {
        edge.push_back(vector<int>());

        last++;
      }

  edge[last].push_back(i);
}

void ggraph::input(const vector<int> &row) {
  int i;

  for (i = 0; i < row.size(); i++)
    input(row[i]);
}

void ggraph::new_row() {
  if (edge.size() == 0)
    throw(string("Warning: Illegal entries!"));

  if (edge.size() >= edge[0].size())
    return;

  if (edge.size() > 1 && edge[0].size() != edge[edge.size() - 1].size())
    throw(string(
        "Warning: Dealing with the new row before the previous is completed!"));

  edge.push_back(vector<int>());
}

void ggraph::output(ostream &ostr) const {
  if (edge.size() == 0)
    throw(string(
        "Warning: Trying to writing down the graph which is not given!"));

  if (edge.size() != edge[edge.size() - 1].size())
    throw(string("Warning: Trying to writing down the graph which is not "
                 "completely given!"));

  int i;
  int j;

  for (i = 0; i < edge.size(); i++) {
    int size = edge[i].size();

    if (i > 0)
      ostr << "\n";

    for (j = 0; j < size; j++) {
      if (j > 0)
        ostr << ' ';

      ostr << edge[i][j];
    }
  }
}

istream &operator>>(istream &istr, ggraph &g) {
  g.input(istr);

  return istr;
}

ostream &operator<<(ostream &ostr, const ggraph &g) {
  g.output(ostr);

  return ostr;
}

class matrix {
  vector<vector<int>> element;

public:
  matrix() {}
  matrix(vector<vector<int>> &);

  int &operator()(const int, const int);
  const vector<int> &row(const int) const;

  void input(istream &);
  void add(const vector<int> &);
};

matrix::matrix(vector<vector<int>> &e) { element = e; }

int &matrix::operator()(const int i, const int j) {
  if (element.size() == 0)
    throw(string("Warning: Access to the element of unformed matrix!"));

  if (element[element.size() - 1].size() != element[0].size())
    throw(string("Warning: Access to the element of unformed matrix!"));

  if (i < 1 || j < 1 || i > element.size() || j > element[0].size())
    throw(string("Warning: The element indices are out of range!"));

  return element[i - 1][j - 1];
}

const vector<int> &matrix::row(const int i) const {
  if (element.size() == 0)
    throw(string("Warning: Access to the row of unformed matrix!"));

  if (element[element.size() - 1].size() != element[0].size())
    throw(string("Warning: Access to the row of unformed matrix!"));

  if (i < 1 || i > element.size())
    throw(string("Warning: The index of the matrix row is out of range!"));

  return element[i - 1];
}

void matrix::input(istream &istr) {
  element = vector<vector<int>>();

  while (true) {
    vector<int> row;

    istr >> row;

    if (row.empty())
      break;

    if (!element.empty() && element[0].size() != row.size())
      throw(string("Warning: Illegal entries!"));

    element.push_back(row);
  }

  if (element.empty())
    throw(string("Warning: There is no matrix in the input file."));
}

void matrix::add(const vector<int> &row) {
  if (!element.empty())
    if (element[0].size() != row.size())
      throw(string("Warning: Illegal entries!"));

  element.push_back(row);
}

istream &operator>>(istream &istr, matrix &m) {
  m.input(istr);

  return istr;
}

void open_to_read(char *name, ifstream &ifstr) {
  string_converter conv(string("workspace/") + name);

  ifstr.open(conv.c_string());

  if (!ifstr)
    throw(string("File ") + string(name) + string(" does not exist!"));
}

bool wait(ifstream &ifstr, char c) {
  char zn;

  do {
    ifstr.get(zn);

    if (ifstr.eof())
      return false;
  } while (zn != c);

  return true;
}

void all_max(symmetric_matrix &c_matrix, symmetric_matrix &n_matrix,
             vector<vector<int>> &good_vertex, double lambda) {
  try {
    ifstream cliques("workspace/extensions/cliques.txt");
    ofstream graphs("maxexts.tmp");

    ggraph c(c_matrix);
    ggraph adjacent(n_matrix);
    matrix goodvert(good_vertex);
    int counter = 0;

    while (wait(cliques, ':')) {
      int i, j;
      ggraph g;
      matrix down_left;
      vector<int> description;

      counter++;
      cliques >> description;

      for (i = 0; i < description.size(); i++)
        down_left.add(goodvert.row(description[i]));

      for (i = 1; i <= c.row(1).size(); i++) {
        if (g.rows() > 0)
          g.new_row();

        g.input(c.row(i));

        for (j = 1; j <= description.size(); j++)
          g.input(down_left(j, i));
      }

      for (i = 0; i < description.size(); i++) {
        if (g.rows() > 0)
          g.new_row();

        g.input(down_left.row(i + 1));

        for (j = 0; j < description.size(); j++)
          g.input(adjacent(description[i], description[j]));
      }

      if (lambda == 0 || lambda == -1) {
        int base = c.rows();
        int total = g.rows();
        vector<bool> included;

        for (int i = 0; i <= total; ++i) {
          included.push_back(true);
        }

        for (int i = 1; i <= base; ++i) {
          for (int j = base + 1; j <= total; ++j) {
            bool found = false;

            for (int k = 1; k <= total; ++k) {
                if (k != i && k != j && g(i, k) != g(j, k)) {
                found = true;

                break;
              }
            }

            if (found == false) {
              included[j] = false;
            }
          }
        }

        ggraph other;

        for (int i = 1; i <= total; ++i) {
          if (included[i]) {
            if (other.rows() > 0) {
              other.new_row();
            }

            for (int j = 1; j <= total; ++j) {
              if (included[j]) {
                other.input(g(i, j));
              }
            }
          }
        }

        g = other;
      }

      graphs << counter << "\n\n" << g << endl << endl;
    }
  } catch (string s) {
  }
}
